<?php
/* --------------------------------------------------------------
  api.php 2019-09-23
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

require_once __DIR__ . '/bootstrap.inc.php';

if(!\StyleEdit\Authentication::isAuthenticated())
{
	\StyleEdit\Authentication::showForbiddenPage();
}

require_once __DIR__ . '/vendor/Slim/Slim.php';

/**
 * GET api.php/languages/:code Get language ($code) texts as an associative array (JSON)
 * GET api.php/templates/:template/config Get configuration values from config.inc.php (the config values are white-listed for security).
 * GET api.php/templates/:template/styles Get style collection
 * GET api.php/templates/:template/styles/:name Get style by name
 * GET api.php/templates/:template/download/:name Download style by name
 * GET api.php/templates/:template/boilerplates Get boilerplate collection
 * POST api.php/logout Log out user
 * POST api.php/templates/:template/styles Create new style
 * POST api.php/templates/:template/styles/:name Duplicate style by source name
 * POST api.php/templates/:template/upload Create by file-upload
 * POST api.php/templates/:template/images Upload new image.
 * PUT api.php/templates/:template/styles/:name Update style by name
 * PATCH api.php/templates/:template/styles/:name/actions/:action Partially update style by name and action
 * DELETE api.php/templates/:template/styles/:name Delete style by name
 * DELETE api.php/templates/:template/images/:name Delete image file.
 */

$styleEditServiceFactory = new StyleEdit\Factories\ServiceFactory();

\Slim\Slim::registerAutoloader();

$config          = array();
$config['mode']  = \StyleEditConfig::DEBUG_BACKTRACE ? 'backtrace' : '';
$config['debug'] = \StyleEditConfig::DEBUG_HTML;

$api = new \Slim\Slim($config);

/**
 * Write Response Function
 *
 * Use this method for returning normal JSON or plain text responses.
 *
 * @param \Slim\Slim $api
 * @param mixed      $response
 * @param bool       $isJsonEncoded       (optional)
 * @param bool       $plainResponseHeader (optional)
 */
$writeResponse = function ($api, $response, $isJsonEncoded = false, $plainResponseHeader = false)
{
	if(!$plainResponseHeader)
	{
		$api->response->headers->set('Content-Type', 'application/json');
	}
	else
	{
		$api->response->headers->set('Content-Type', 'text/plain'); // Needed for Internet Explorer Compatibility
	}

	if(!$isJsonEncoded)
	{
		$response = \StyleEdit\JsonTransformer::encode($response);
	}

	$api->response->write($response);
};

/**
 * Write Style Response Function
 *
 * Use this method for returning specific style-formatted responses. The template-translations will be also included in
 * the response. The template-translations can be extended by additional translations files in the format
 * [language_code].[any_string].json. They will be merged into the standard template-translations.
 *
 * @param \Slim\Slim $api
 * @param mixed      $response
 * @param string     $template
 * @param bool       $isJsonEncoded
 */
$writeStyleResponse = function ($api, $response, $template, $isJsonEncoded = false) use ($writeResponse)
{
	$response = array('config' => $response);

	$lang = \StyleEditConfig::DEFAULT_LANGUAGE;
	if($api->request->get('lang') !== null && preg_match('/^[a-z]{2}$/', $api->request->get('lang')))
	{
		$lang = $api->request->get('lang');
	}

	$filePath = \StyleEdit\StyleThemeControl::getStylesDirectoryPath($template) . 'lang/' . $lang . '.json';

	if(file_exists($filePath))
	{
		$response['lang'] = \StyleEdit\JsonTransformer::decode(file_get_contents($filePath));
	}
	
	$usermodfilePath = \StyleEdit\StyleThemeControl::getStylesDirectoryPath($template) . 'lang/' . $lang . '.*.json';
	$files = glob($usermodfilePath);
	
	if($files === false) {
	    $files = [];
    }
	
	if(\StyleEditConfig::getAdditionalBoxesBasePath() !== null)
	{
        $rglob = function($pattern) use (&$rglob)
        {
            $files = glob($pattern);
            foreach (glob(dirname($pattern) . '/*', GLOB_ONLYDIR | GLOB_NOSORT) as $dir) {
                $dir = str_replace('\\', '/', $dir);
                if (strpos($dir, '/node_modules') === false) {
                    $files = array_merge($files, $rglob($dir . '/' . basename($pattern)));
                }
            }
            
            return $files;
        };
	    
        $dirContent   = $rglob(\StyleEditConfig::getAdditionalBoxesBasePath() . '*');
        $boxesLangDir = [];
		
		foreach($dirContent as $file)
		{
			$boxesAll         = \StyleEditConfig::getAdditionalBoxesDirectoryPath() . '/Lang/' . $lang . '.';
			$boxesTemplate    = \StyleEditConfig::getAdditionalBoxesDirectoryPath($template) . '/Lang/' . $lang . '.';
			$hasBoxesAll      = stripos($file, $boxesAll);
			$hasBoxesTemplate = stripos($file, $boxesTemplate);
			
			if(($hasBoxesAll !== false || $hasBoxesTemplate !== false) && substr($file, -5) === '.json')
			{
				$boxesLangDir[] = $file;
			}
		}
		
		$files = array_merge($files, $boxesLangDir);
	}
	
	if(is_array($files))
	{
		foreach($files as $file)
		{
			$langArray = \StyleEdit\JsonTransformer::decode(file_get_contents($file));
			$response['lang'] = array_merge($response['lang'], $langArray);
		}
	}
	
	$response['images'] = array();
	if(file_exists(\StyleEditConfig::getImagesDirectoryPath($template)))
	{
		$imageHandler = new \StyleEdit\ImageHandler(\StyleEditConfig::getImagesDirectoryPath($template));
		foreach($imageHandler->getImageFilenames() as $filename)
		{
			$response['images'][] = $filename;
		}
	}

	$writeResponse($api, $response, $isJsonEncoded);
};

/**
 * Get language ($code) texts as an associative array (JSON)
 */
$api->get('/languages/:code', function ($code) use ($api, $writeResponse)
{
	$response = '{}';
	$filePath = __DIR__ . '/lang/' . $code . '.json';

	if(file_exists($filePath))
	{
		$response = file_get_contents($filePath);
	}

	$writeResponse($api, $response, $isJsonEncoded = true);
})->conditions(array('code' => '[a-z]{2}'));

$api->group('/templates/:template',
	function () use ($api, $styleEditServiceFactory, $writeResponse, $writeStyleResponse)
	{
		$api->get('/config', function ($template) use ($api, $writeResponse)
		{
			$response = array(
				'defaultLanguageCode' => \StyleEditConfig::DEFAULT_LANGUAGE,
				'imagesDir' => \StyleEditConfig::getImagesDirectory($template)
			);
			
			$writeResponse($api, $response, false);
		});
		
		$api->get('/styles', function ($template) use ($api, $styleEditServiceFactory, $writeStyleResponse)
		{
			$styleEditReadService = $styleEditServiceFactory->createReadService($template);
			$stylesArray          = $styleEditReadService->getStyleConfigs()->toArray();

			$response = array();

			foreach($stylesArray as $styleConfig)
			{
				$response[] = $styleConfig->getJsonDataArray();
			}

			$writeStyleResponse($api, $response, $template);
		});

		$api->get('/styles/:name', function ($template, $name) use ($api, $styleEditServiceFactory, $writeStyleResponse)
		{
			$styleEditReadService = $styleEditServiceFactory->createReadService($template);
			$styleConfig          = $styleEditReadService->getStyleConfigByName(\StyleEdit\Utf8Converter::encode($name));

			$writeStyleResponse($api, $styleConfig->getJsonDataArray(), $template);
		});

		$api->get('/download/:name', function ($template, $name) use ($api, $styleEditServiceFactory, $writeResponse)
		{
			$styleEditReadService = $styleEditServiceFactory->createReadService($template);
			$styleConfig          = $styleEditReadService->getStyleConfigByName(\StyleEdit\Utf8Converter::encode($name));
			$filename             = $styleConfig->getFilename();
			$styleConfigArray     = $styleConfig->getJsonDataArray();

			$responseJsonString = \StyleEdit\JsonTransformer::encode($styleConfigArray);

			// Using Slim-Framework-Headers does not work in Chrome (filename will be ignored).
			// So let's do it the old way...

			header('Pragma: public');
			header('Expires: 0');
			header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
			header('Content-Description: File Transfer');
			header('Content-Type: Application/octet-stream');
			header('Content-Disposition: attachment; filename="' . $filename . '"');
			header('Content-Transfer-Encoding: binary');
			header('Cache-Control: max-age=0');
			header('Content-Length: ' . strlen($responseJsonString));

			echo $responseJsonString;

			// Without this command Safari will add a ".html" extension to the file.
			// http://stackoverflow.com/a/22461099
			exit;
		});

		$api->get('/boilerplates', function ($template) use ($api, $styleEditServiceFactory, $writeStyleResponse)
		{
			$styleEditReadService = $styleEditServiceFactory->createReadService($template);
			$boilerplatesArray    = $styleEditReadService->getBoilerplates()->toArray();
			$response             = array();

			foreach($boilerplatesArray as $styleConfig)
			{
				$response[] = $styleConfig->getJsonDataArray();
			}

			$writeStyleResponse($api, $response, $template);
		});

		$api->post('/styles', function ($template) use ($api, $styleEditServiceFactory, $writeResponse)
		{
			$styleEditWriteService = $styleEditServiceFactory->createWriteService($template);
			$newStoredStyleConfig  = $styleEditWriteService->createStyleConfig($api->request->post('boilerplate_name'),
			                                                                   $api->request->post('new_style_name'));

			$writeResponse($api, $newStoredStyleConfig->getJsonDataArray());
		});

		$api->post('/styles/:name', function ($template, $name) use ($api, $styleEditServiceFactory, $writeResponse)
		{
			$styleEditWriteService = $styleEditServiceFactory->createWriteService($template);
			$newStoredStyleConfig  = $styleEditWriteService->copyStyleConfig(\StyleEdit\Utf8Converter::encode($name),
			                                                                 $api->request->post('new_style_name'));

			$writeResponse($api, $newStoredStyleConfig->getJsonDataArray());
		});

		$api->post('/upload', function ($template) use ($api, $styleEditServiceFactory, $writeResponse)
		{
			if(isset($_FILES['json-upload']))
			{
				$upload =& $_FILES['json-upload'];

				if($upload['error'] === UPLOAD_ERR_OK
				   && is_uploaded_file($upload['tmp_name'])
				   && strlen($upload['name']) > 5
				   && substr($upload['name'], -5) === '.json'
				)
				{
					$uploadedJsonFileContent = file_get_contents($_FILES['json-upload']['tmp_name']);

					$styleEditWriteService = $styleEditServiceFactory->createWriteService($template);
					$newStoredStyleConfig  = $styleEditWriteService->uploadStyleConfig($api->request->post('style_name'),
					                                                                   $uploadedJsonFileContent);

					$plainResponseHeader = (strpos($_SERVER['HTTP_USER_AGENT'], 'Trident') !== false
					                        || strpos($_SERVER['HTTP_USER_AGENT'], 'Edge') !== false);

					$writeResponse($api, $newStoredStyleConfig->getJsonDataArray(), false, $plainResponseHeader);
				}
			}
		});

		$api->post('/images', function ($template) use ($api, $writeResponse)
		{
			if(\StyleEdit\StyleThemeControl::isThemeSystemActive())
			{
				$imageHandler = new \StyleEdit\ImageHandler(\StyleEdit\StyleThemeControl::getOriginalThemeImagesDirectoryPath($template));
				
				$imageHandler->copy($_FILES['image']);
			}
			
			$imageHandler = new \StyleEdit\ImageHandler(\StyleEditConfig::getImagesDirectoryPath($template));

			$imageHandler->move($_FILES['image']);

			$writeResponse($api, preg_replace("/[^A-Za-z0-9\_\-\.]/", '-', $_FILES['image']['name']), false, true);
		});

		$api->put('/styles/:name', function ($template, $name) use ($api, $styleEditServiceFactory, $writeResponse)
		{
			$styleEditWriteService = $styleEditServiceFactory->createWriteService($template);
			$styleConfig           = $styleEditWriteService->updateStyleConfig(\StyleEdit\Utf8Converter::encode($name),
			                                                                   $api->request->post('style_json'));

			$writeResponse($api, $styleConfig->getJsonDataArray());
		});

		$api->patch('/styles/:name/actions/:action',
			function ($template, $name, $action) use ($api, $styleEditServiceFactory, $writeResponse)
			{
				$response = [];

				if($action === 'activate')
				{
					$styleEditWriteService = $styleEditServiceFactory->createWriteService($template);
					$styleEditWriteService->activateStyle(\StyleEdit\Utf8Converter::encode($name));

					$styleEditReadService = $styleEditServiceFactory->createReadService($template, false);
					$stylesArray          = $styleEditReadService->getStyleConfigs()->toArray();

					$response = array();

					foreach($stylesArray as $styleConfig)
					{
						$response[] = $styleConfig->getJsonDataArray();
					}
				}

				$writeResponse($api, $response);
			});

		$api->delete('/styles/:name', function ($template, $name) use ($api, $styleEditServiceFactory, $writeResponse)
		{
			$styleEditWriteService = $styleEditServiceFactory->createWriteService($template);
			$styleEditWriteService->deleteStyleConfigByName(\StyleEdit\Utf8Converter::encode($name));
			
			$styleEditReadService = $styleEditServiceFactory->createReadService($template, false);
			$stylesArray          = $styleEditReadService->getStyleConfigs()->toArray();

			$response = array();

			foreach($stylesArray as $styleConfig)
			{
				$response[] = $styleConfig->getJsonDataArray();
			}

			$writeResponse($api, $response);
		});

		$api->delete('/images/:image', function ($template, $image) use ($api, $writeResponse)
		{
			if(\StyleEdit\StyleThemeControl::isThemeSystemActive())
			{
				$imageHandler = new \StyleEdit\ImageHandler(\StyleEdit\StyleThemeControl::getOriginalThemeImagesDirectoryPath($template));
				
				$imageHandler->delete($image);
			}
			
			$imageHandler = new \StyleEdit\ImageHandler(\StyleEditConfig::getImagesDirectoryPath($template));

			$imageHandler->delete($image);

			$writeResponse($api, array('filename' => $image));
		});
	});

$api->post('/logout', function () use ($api, $writeResponse)
{
	\StyleEdit\Authentication::setAuthenticationToInvalid();

	if(file_exists(\StyleEditConfig::getCssCacheFilePath()))
	{
		unlink(\StyleEditConfig::getCssCacheFilePath());
	}
	
	if(\StyleEdit\StyleThemeControl::isThemeSystemActive() && is_dir(__DIR__  . '/../public/theme/styles/styleedit'))
	{
		foreach(new DirectoryIterator(__DIR__  . '/../public/theme/styles/styleedit') as $templateFile)
		{
			if($templateFile->isDir() || $templateFile->getExtension() !== 'json')
			{
				continue;
			}
			
			if(0 === strpos($templateFile->getFilename(), '__temporary_style_config_'))
			{
				unlink($templateFile->getPathname());
			}
		}
	}
	
	foreach(new DirectoryIterator(__DIR__ . '/templates') as $template)
	{
		if(!$template->isDot() && $template->isDir())
		{
			foreach(new DirectoryIterator($template->getPathname()) as $templateFile)
			{
				if($templateFile->isDir() || $templateFile->getExtension() !== 'json')
				{
					continue;
				}
				
				if(0 === strpos($templateFile->getFilename(), '__temporary_style_config_'))
				{
					unlink($templateFile->getPathname());
				}
			}
		}
	}

	$response = array('status' => 'success');

	$writeResponse($api, $response);
});

/**
 * Error Handling
 */
$api->error(function (\Exception $e) use ($api)
{
	$api->response->setStatus(500);
	$api->response->headers->set('Content-Type', 'application/json');

	$response = array(
		'code'    => $e->getCode(),
		'status'  => 'error',
		'message' => $e->getMessage(),
		'request' => array(
			'method' => $api->request->getMethod(),
			'url'    => $api->request->getUrl(),
			'path'   => $api->request->getPath(),
			'uri'    => array(
				'root'     => $api->request->getRootUri(),
				'resource' => $api->request->getResourceUri()
			)
		)
	);

	// Include full backtrace in response
	if($api->config('mode') === 'backtrace')
	{
		$response['error'] = array(
			'file'  => $e->getFile(),
			'line'  => $e->getLine(),
			'stack' => $e->getTrace()
		);
	}

	$responseJsonString = \StyleEdit\JsonTransformer::encode($response);

	$api->response->write($responseJsonString);
});

$api->run();
